package io.github.kosmx.emotes.common.network;

import com.zigythebird.playeranimcore.animation.Animation;
import io.github.kosmx.emotes.common.CommonData;
import io.github.kosmx.emotes.common.network.objects.*;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import java.io.IOException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Send everything emotes mod data...
 */
public class EmotePacket {
    public static final HashMap<Byte, Byte> defaultVersions = new HashMap<>();

    static {
        AbstractNetworkPacket tmp = new NewAnimPacket();
        defaultVersions.put(tmp.getID(), tmp.getVer());
        tmp = new EmoteDataPacket();
        defaultVersions.put(tmp.getID(), tmp.getVer());
        tmp = new PlayerDataPacket();
        defaultVersions.put(tmp.getID(), tmp.getVer());
        tmp = new DiscoveryPacket();
        defaultVersions.put(tmp.getID(), tmp.getVer());
        tmp = new StopPacket();
        defaultVersions.put(tmp.getID(), tmp.getVer());
        tmp = new SongPacket();
        defaultVersions.put(tmp.getID(), tmp.getVer());
        tmp = new EmoteHeaderPacket();
        defaultVersions.put(tmp.getID(), tmp.getVer());
        tmp = new EmoteIconPacket();
        defaultVersions.put(tmp.getID(), tmp.getVer());
    }

    public final NetHashMap subPackets = new NetHashMap();

    public final NetData data;

    int version;

    protected EmotePacket(@NotNull NetData data) {
        //Make sure every packet has a version...
        if (data.versions == null || data.versions.isEmpty()) {
            data.versions = new HashMap<>(defaultVersions);
        } /*else {
            defaultVersions.forEach((aByte, bByte) -> {
                if(!data.versions.containsKey(aByte)){
                    data.versions.put(aByte, bByte);
                }
            });
        }*/

        this.data = data;
        subPackets.put(new NewAnimPacket());
        subPackets.put(new EmoteDataPacket());
        subPackets.put(new PlayerDataPacket());
        subPackets.put(new StopPacket());
        subPackets.put(new DiscoveryPacket());
        subPackets.put(new SongPacket());
        subPackets.put(new EmoteHeaderPacket());
        subPackets.put(new EmoteIconPacket());
    }

    //Write packet to a new ByteBuf
    public ByteBuffer write() throws IOException {
        if(data.purpose == PacketTask.UNKNOWN)throw new IllegalArgumentException("Can't send packet without any purpose...");
        AtomicReference<Byte> partCount = new AtomicReference<>((byte) 0);
        AtomicInteger sizeSum = new AtomicInteger(6); //5 bytes is the header
        subPackets.forEach((aByte, packet) -> {
            if(packet.doWrite(this.data)){
                if(!(packet instanceof SongPacket)){
                    partCount.getAndSet((byte) (partCount.get() + 1));
                    sizeSum.addAndGet(packet.calculateSize(this.data) + 6); //it's size + the header
                }
            }
        });
        if(data.strictSizeLimit && sizeSum.get() > data.sizeLimit)throw new IOException(String.format(
                "Can't send emote, packet's size (%s) is bigger than max allowed (%s)!", sizeSum.get(), data.sizeLimit
        ));
        SongPacket songPacket = (SongPacket) subPackets.get((byte)3);
        int songSize = songPacket.calculateSize(this.data) + 6;
        if(songPacket.doWrite(this.data) && sizeSum.get() + songSize <= data.sizeLimit){
            partCount.getAndSet((byte) (partCount.get() + 1));
            sizeSum.addAndGet(songSize);
        }
        else data.writeSong = false;

        ByteBuffer buf = ByteBuffer.allocate(sizeSum.get());

        buf.putInt(subPackets.get((byte)8).getVer(data.versions));
        buf.put(data.purpose.id);
        buf.put(partCount.get());

        try {
            for (AbstractNetworkPacket packet : this.subPackets.values()) {
                writeSubPacket(buf, packet);
            }
        } catch (Throwable th) {
            throw new IOException("Exception while writing sub-packages", th);
        } finally {
            ((Buffer)buf).flip(); // make it ready to read
        }

        return buf;
    }

    void writeSubPacket(ByteBuffer byteBuffer, AbstractNetworkPacket packetSender) throws IOException {
        if(packetSender.doWrite(this.data)){
            //This is not time critical task, HeapByteBuf is more secure and I can wrap it again.
            int len = packetSender.calculateSize(this.data);
            byteBuffer.put(packetSender.getID());
            byteBuffer.put(packetSender.getVer(data.versions));
            byteBuffer.putInt(len);
            int currentIndex = byteBuffer.position();
            packetSender.write(byteBuffer, this.data);
            if(byteBuffer.position() != currentIndex + len){
                throw new IOException(String.format("Incorrect size calculator: %s (calculated %s, real %s)",
                        packetSender.getClass(), len, byteBuffer.position() - currentIndex
                ));
            }
        }
    }

    @NotNull
    public NetData read(ByteBuffer byteBuffer) throws IOException {
        this.version = byteBuffer.getInt();
        if (this.version > CommonData.networkingVersion) throw new IOException("Can't read newer version");
        data.purpose = PacketTask.getTaskFromID(byteBuffer.get());

        byte count = byteBuffer.get();

        for (int i = 0; i < count; i++) {
            byte id = byteBuffer.get();
            byte sub_version = byteBuffer.get();
            int size = byteBuffer.getInt();
            int currentPos = byteBuffer.position();

            if (subPackets.containsKey(id)) {
                try {
                    subPackets.get(id).read(byteBuffer, this.data, sub_version);
                } catch (Throwable th) {
                    throw new IOException("Invalid " + subPackets.get(id).getClass().getName() + " sub-packet received", th);
                }

                if (byteBuffer.position() != size + currentPos) {
                    ((Buffer)byteBuffer).position(currentPos + size);
                }
            } else {
                ((Buffer)byteBuffer).position(currentPos + size);
            }
        }

        if (data.prepareAndValidate()) return this.data;
        else throw new IOException("no valid data");
    }

    /**
     * EmotePacket builder.
     */
    public static class Builder{

        final NetData data;
        /**
         * To send an emote
         */
        public Builder setVersion(HashMap<Byte, Byte> versions){
            data.versions = versions;
            return this;
        }

        public NetData copyAndGetData(){
            return data.copy();
        }

        public Builder(NetData data){
            this.data = data;
        }

        public Builder copy(){
            return new Builder(this.data.copy());
        }

        public Builder(){
            data = new NetData();
        }

        public Builder setThreshold(float t){
            data.threshold = t;
            return this;
        }

        public EmotePacket build(){
            return new EmotePacket(data);
        }

        public EmotePacket build(int sizeLimit, boolean strict){
            return this.setSizeLimit(sizeLimit, strict).build();
        }

        public Builder setSizeLimit(int sizeLimit, boolean strict){
            if(sizeLimit <= 0)throw new IllegalArgumentException("Size limit must be positive");
            data.sizeLimit = sizeLimit;
            data.strictSizeLimit = strict;
            return this;
        }

        public Builder configureToStreamEmote(Animation emoteData, @Nullable UUID player){
            if(data.purpose != PacketTask.UNKNOWN)throw new IllegalArgumentException("Can's send and stop emote at the same time");
            data.purpose = PacketTask.STREAM;
            data.emoteData = emoteData;
            data.player = player;
            return this;
        }

        public Builder configureToSaveEmote(Animation emoteData){
            if(data.purpose != PacketTask.UNKNOWN)throw new IllegalArgumentException("already configured?!");
            data.purpose = PacketTask.FILE;
            data.sizeLimit = Integer.MAX_VALUE;
            data.emoteData = emoteData;
            return this;
        }

        public Builder configureEmoteTick(float tick) {
            this.data.tick = tick;
            return this;
        }

        public Builder configureTarget(@Nullable UUID target){
            data.player = target;
            return this;
        }

        public Builder configureToStreamEmote(Animation emoteData) {
            return configureToStreamEmote(emoteData, null);
        }

        public Builder configureToSendStop(UUID emoteID, @Nullable UUID player){
            if(data.purpose != PacketTask.UNKNOWN)throw new IllegalArgumentException("Can't send emote and stop at the same time");
            data.purpose = PacketTask.STOP;
            data.stopEmoteID = emoteID;
            data.player = player;
            return this;
        }

        public Builder configureToSendStop(UUID emoteID){
            return configureToSendStop(emoteID, null);
        }

        public Builder configureToConfigExchange(boolean songEnabled){
            if(data.purpose != PacketTask.UNKNOWN)throw new IllegalArgumentException("Can't send config with emote or stop data...");
            data.purpose = PacketTask.CONFIG;
            HashMap<Byte, Byte> versions = new HashMap<>(EmotePacket.defaultVersions);
            if(!songEnabled){
                versions.replace((byte)3, (byte)0);
            }
            this.data.versions = versions;
            return this;
        }

        public void removePlayerID(){
            this.data.player = null;
        }

        public Builder strictSizeLimit(boolean strict) {
            data.strictSizeLimit = strict;
            return this;
        }
    }
}
