package dev.kosmx.playerAnim.minecraftApi;

import dev.kosmx.playerAnim.api.IPlayable;
import dev.kosmx.playerAnim.minecraftApi.codec.AnimationCodecs;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_5250;
import net.minecraft.class_5455;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

/**
 * Load resources from <code>assets/{modid}/player_animation</code>
 * <br>
 * The animation identifier:
 * <table border="1">
 *   <tr>
 *     <td> namespace </td> <td> Mod namespace </td>
 *   </tr>
 *   <tr>
 *     <td> path </td> <td> Animation name, not the filename </td>
 *   </tr>
 * </table>
 * <br>
 * Use {@link PlayerAnimationRegistry#getAnimation(class_2960)} to fetch an animation
 * <br><br>
 * Extra animations can be added by ResourcePack(s) or other mods
 * <br><br>
 * Breaking change with 2.0.0: Registry now returns an IPlayable type instead of a KeyframeAnimation.
 * Feel free to safely cast it into KeyframeAnimation:
 * <br>
 * <code>if (PlayerAnimationRegistry.getAnimation(id) instanceof KeyframeAnimation animation) {...}</code>
 * <br>
 * Or to simply play the result, do<br>
 * <code>PlayerAnimationRegistry.getAnimation(id).playAnimation()</code> <br>
 * This change will allow more animation formats to be supported. (Don't forget, you can still wrap the unknown animation in custom wrappers :D)
 */
@Environment(EnvType.CLIENT)
public final class PlayerAnimationRegistry {

    private static final HashMap<class_2960, IPlayable> animations = new HashMap<>();
    private static final Logger logger = LoggerFactory.getLogger(PlayerAnimationRegistry.class);

    /**
     * Get an animation from the registry, using Identifier(MODID, animation_name) as key
     * @param identifier identifier
     * @return animation, <code>null</code> if no animation
     */
    @Nullable
    public static IPlayable getAnimation(@NotNull class_2960 identifier) {
        return animations.get(identifier);
    }

    /**
     * Get Optional animation from registry
     * @param identifier identifier
     * @return Optional animation
     */
    @NotNull
    public static Optional<IPlayable> getAnimationOptional(@NotNull class_2960 identifier) {
        return Optional.ofNullable(getAnimation(identifier));
    }

    /**
     * @return an unmodifiable map of all the animations
     */
    public static Map<class_2960, IPlayable> getAnimations() {
        return Map.copyOf(animations);
    }

    /**
     * Returns the animations of a specific mod/namespace
     * @param modid namespace (assets/modid)
     * @return map of path and animations
     */
    @NotNull
    public static Map<String, IPlayable> getModAnimations(@NotNull String modid) {
        HashMap<String, IPlayable> map = new HashMap<>();
        for (Map.Entry<class_2960, IPlayable> entry: animations.entrySet()) {
            if (entry.getKey().method_12836().equals(modid)) {
                map.put(entry.getKey().method_12832(), entry.getValue());
            }
        }
        return map;
    }

    /**
     * Load animations using ResourceManager
     * Internal use only!
     */
    @ApiStatus.Internal
    public static void resourceLoaderCallback(@NotNull class_3300 manager) {
        animations.clear();

        for (var resource: manager.method_14488("player_animations", ignore -> true).entrySet()) {
            var extension = AnimationCodecs.getExtension(resource.getKey().method_12832());
            if (extension == null) continue;
            var a = AnimationCodecs.deserialize(extension, () -> {
                try {
                    return resource.getValue().method_14482();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            });
            for(var animation: a) {
                try {
                    animations.put(class_2960.method_60655(resource.getKey().method_12836(), serializeTextToString(animation.getName())), animation);
                } catch (Throwable e) {
                    logger.debug("Failed to load animation with name space {} and name {}. Either a PAL animation or has an invalid name.", resource.getKey().method_12836(), animation.getName());
                }
            }
        }
        for (var resource: manager.method_14488("player_animation", ignore -> true).entrySet()) {
            var extension = AnimationCodecs.getExtension(resource.getKey().method_12832());
            if (extension == null) continue;
            logger.warn("[WARNING FOR MOD DEVS] Animation {} is in wrong directory: \"player_animation\", please place it in \"player_animations\".", resource.getKey().method_12832());
            var a = AnimationCodecs.deserialize(extension, () -> {
                try {
                    return resource.getValue().method_14482();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            });
            for(var animation: a) {
                animations.put(class_2960.method_60655(resource.getKey().method_12836(), serializeTextToString(animation.getName())), animation);
            }
        }
    }


    /**
     * Helper function to convert animation name to string
     */
    public static String serializeTextToString(String arg) {
        try {
            var component = class_2561.class_2562.method_10877(arg, class_5455.field_40585);
            if (component != null) {
                return component.getString();
            }
        } catch(Exception ignored) { }
        return arg.replace("\"", "");
    }
}
